import assign from 'lodash.assign';
import classnames from 'classnames/dedupe'

const { addFilter } = wp.hooks;
const { __ } = wp.i18n;

const classPrefix = 'has-'

// Enable Animation control on the following blocks
const enableAnimation = [
    'core/columns',
    'core/group',
    'core/cover',
];

const StyleValues = [{
    label: __('Not animated', 'thor'),
    value: 'sans',
  },
  {
    label: __('Fade-in bottom to top', 'thor'),
    value: 'fade-bt',
  },
];

/**
 * Add Animation control attribute to block.
 *
 * @param {object} settings Current block settings.
 * @param {string} name Name of block.
 *
 * @returns {object} Modified block settings.
 */
const AnimationControl = (settings, name) => {
    // Do nothing if it's another block than our defined ones.
    if (!enableAnimation.includes(name)) {
        return settings;
    }

    // Use Lodash's assign to gracefully handle if attributes are undefined
    settings.attributes = assign(settings.attributes, {
        Style: {
            type: 'string',
            default: '',
        },
    });
    return settings;

};

addFilter('blocks.registerBlockType', 'register-block-control/attribute/AnimationControl', AnimationControl);

const { createHigherOrderComponent } = wp.compose;
const { Fragment } = wp.element;
const { InspectorControls } = wp.blockEditor;
const { PanelBody, SelectControl } = wp.components;

/**
 * Create HOC to add Animation control to inspector controls of block.
 */
const extendAnimationControl = createHigherOrderComponent((BlockEdit) => {
    return ( props ) => {
        // Do nothing if it's another block than our defined ones.
        if ( ! enableAnimation.includes( props.name ) ) {
            return (
                <BlockEdit { ...props } />
                );
            }
            
        const { Style } = props.attributes;

        let classNames = props.attributes.className

        StyleValues.forEach(({ value}) => {
          classNames = classnames(classNames, { [`${classPrefix}${value}-animation`]: Style == value })
        })

        props.attributes.className = classNames


        return (
            <Fragment>
                <BlockEdit {...props} />
                <InspectorControls>
                    <PanelBody
                        title={__('Animation', 'thor')}
                        initialOpen={true}
                    >
                        <SelectControl
                            label={__('Select animation', 'thor')}
                            value={Style}
                            options={StyleValues}
                            onChange={(selectedStyleOption) => {
                                props.setAttributes({
                                    Style: selectedStyleOption,
                                });
                            }}
                        />
                    </PanelBody>
                </InspectorControls>
            </Fragment>
        );
    };
}, 'extendAnimationControl');

addFilter('editor.BlockEdit', 'extend-block-example/with-Animation-control', extendAnimationControl);



/**
 * Add margin style attribute to save element of block.
 *
 * @param {object} saveElementProps Props of save element.
 * @param {Object} blockType Block type information.
 * @param {Object} attributes Attributes of block.
 *
 * @returns {object} Modified props of save element.
 */
const styleAnimationControl = (saveElementProps, blockType, attributes) => {
    // Do nothing if it's another block than our defined ones.
    if (!enableAnimation.includes(blockType.name)) {
        return saveElementProps;
    }

    return saveElementProps

};

addFilter(
    'blocks.getSaveContent.extraProps',
    'register-block-control/get-save-content/extra-props',
    styleAnimationControl
);