import assign from 'lodash.assign';
import classnames from 'classnames/dedupe'

const { addFilter } = wp.hooks;
const { __ } = wp.i18n;

const classPrefix = 'is-'

// Available Weight control options
const mobileValues = [
    {
        label: __('Show', 'thor'),
        value: 'mobile-show',
    },
    {
        label: __('Hide', 'thor'),
        value: 'mobile-hide',
    },
];

const tabletValues = [
    {
        label: __('Show', 'thor'),
        value: 'tablet-show',
    },
    {
        label: __('Hide', 'thor'),
        value: 'tablet-hide',
    },
];

const desktopValues = [
    {
        label: __('Show', 'thor'),
        value: 'desktop-show',
    },
    {
        label: __('Hide', 'thor'),
        value: 'desktop-hide',
    },
];

const wideValues = [
    {
        label: __('Show', 'thor'),
        value: 'wide-show',
    },
    {
        label: __('Hide', 'thor'),
        value: 'wide-hide',
    },
];

/**
 * Add Weight control attribute to block.
 *
 * @param {object} settings Current block settings.
 * @param {string} name Name of block.
 *
 * @returns {object} Modified block settings.
 */
const WeightControl = (settings, name) => {

    // Use Lodash's assign to gracefully handle if attributes are undefined
    settings.attributes = assign(settings.attributes, {
        mobile: {
            type: 'string',
            default: '',
        },
        tablet: {
            type: 'string',
            default: '',
        },
        desktop: {
            type: 'string',
            default: '',
        },
        wide: {
            type: 'string',
            default: '',
        },
    });
    return settings;

};

addFilter('blocks.registerBlockType', 'register-block-control/attribute/WeightControl', WeightControl);

const { createHigherOrderComponent } = wp.compose;
const { Fragment } = wp.element;
const { InspectorControls } = wp.blockEditor;
const { PanelBody, SelectControl } = wp.components;

/**
 * Create HOC to add Weight control to inspector controls of block.
 */
const extendWeightControl = createHigherOrderComponent((BlockEdit) => {
    return ( props ) => {

        const { mobile } = props.attributes;
        const { tablet } = props.attributes;
        const { desktop } = props.attributes;
        const { wide } = props.attributes;

        let classNames = props.attributes.className

        mobileValues.forEach(({ value}) => {
          classNames = classnames(classNames, { [`${classPrefix}${value}`]: mobile == value })
        })

        tabletValues.forEach(({ value}) => {
          classNames = classnames(classNames, { [`${classPrefix}${value}`]: tablet == value })
        })

        desktopValues.forEach(({ value}) => {
          classNames = classnames(classNames, { [`${classPrefix}${value}`]: desktop == value })
        })

        wideValues.forEach(({ value}) => {
          classNames = classnames(classNames, { [`${classPrefix}${value}`]: wide == value })
        })

        props.attributes.className = classNames


        return (
            <Fragment>
                <BlockEdit {...props} />
                <InspectorControls>
                    <PanelBody
                        title={__('Responsive Manager', 'thor')}
                        initialOpen={true}
                    >
                        <SelectControl
                            label={__('Mobile', 'thor')}
                            value={mobile}
                            options={mobileValues}
                            onChange={(selectedmobileOption) => {
                                props.setAttributes({
                                    mobile: selectedmobileOption,
                                });
                            }}
                        />
                        <SelectControl
                            label={__('Tablet', 'thor')}
                            value={tablet}
                            options={tabletValues}
                            onChange={(selectedtabletOption) => {
                                props.setAttributes({
                                    tablet: selectedtabletOption,
                                });
                            }}
                        />
                        <SelectControl
                            label={__('Desktop', 'thor')}
                            value={desktop}
                            options={desktopValues}
                            onChange={(selecteddesktopOption) => {
                                props.setAttributes({
                                    desktop: selecteddesktopOption,
                                });
                            }}
                        />
                        <SelectControl
                            label={__('Wide', 'thor')}
                            value={wide}
                            options={wideValues}
                            onChange={(selectedwideOption) => {
                                props.setAttributes({
                                    wide: selectedwideOption,
                                });
                            }}
                        />
                    </PanelBody>
                </InspectorControls>
            </Fragment>
        );
    };
}, 'extendWeightControl');

addFilter('editor.BlockEdit', 'extend-block-example/with-Weight-control', extendWeightControl);



/**
 * Add margin style attribute to save element of block.
 *
 * @param {object} saveElementProps Props of save element.
 * @param {Object} blockType Block type information.
 * @param {Object} attributes Attributes of block.
 *
 * @returns {object} Modified props of save element.
 */
const styleWeightControl = (saveElementProps, blockType, attributes) => {
    return saveElementProps
};

addFilter(
    'blocks.getSaveContent.extraProps',
    'register-block-control/get-save-content/extra-props',
    styleWeightControl
);