import assign from 'lodash.assign';
import classnames from 'classnames/dedupe'

const { addFilter } = wp.hooks;
const { __ } = wp.i18n;

const classPrefix = 'has-'

// Enable Weight control on the following blocks
const enableWeight = [
    'core/paragraph',
    'core/heading',
    'core/button',
];

const StyleValues = [{
    label: __('Font Sans', 'thor'),
    value: 'sans',
  },
  {
    label: __('Font Mono', 'thor'),
    value: 'mono',
  },
];

/**
 * Add Weight control attribute to block.
 *
 * @param {object} settings Current block settings.
 * @param {string} name Name of block.
 *
 * @returns {object} Modified block settings.
 */
const WeightControl = (settings, name) => {
    // Do nothing if it's another block than our defined ones.
    if (!enableWeight.includes(name)) {
        return settings;
    }

    // Use Lodash's assign to gracefully handle if attributes are undefined
    settings.attributes = assign(settings.attributes, {
        Style: {
            type: 'string',
            default: '',
        },
    });
    return settings;

};

addFilter('blocks.registerBlockType', 'register-block-control/attribute/WeightControl', WeightControl);

const { createHigherOrderComponent } = wp.compose;
const { Fragment } = wp.element;
const { InspectorControls } = wp.blockEditor;
const { PanelBody, SelectControl } = wp.components;

/**
 * Create HOC to add Weight control to inspector controls of block.
 */
const extendWeightControl = createHigherOrderComponent((BlockEdit) => {
    return ( props ) => {
        // Do nothing if it's another block than our defined ones.
        if ( ! enableWeight.includes( props.name ) ) {
            return (
                <BlockEdit { ...props } />
                );
            }
            
        const { Style } = props.attributes;

        let classNames = props.attributes.className

        StyleValues.forEach(({ value}) => {
          classNames = classnames(classNames, { [`${classPrefix}${value}-font-family`]: Style == value })
        })

        props.attributes.className = classNames


        return (
            <Fragment>
                <BlockEdit {...props} />
                <InspectorControls>
                    <PanelBody
                        title={__('Font options', 'thor')}
                        initialOpen={true}
                    >
                        <SelectControl
                            label={__('Font type', 'thor')}
                            value={Style}
                            options={StyleValues}
                            onChange={(selectedStyleOption) => {
                                props.setAttributes({
                                    Style: selectedStyleOption,
                                });
                            }}
                        />
                    </PanelBody>
                </InspectorControls>
            </Fragment>
        );
    };
}, 'extendWeightControl');

addFilter('editor.BlockEdit', 'extend-block-example/with-Weight-control', extendWeightControl);



/**
 * Add margin style attribute to save element of block.
 *
 * @param {object} saveElementProps Props of save element.
 * @param {Object} blockType Block type information.
 * @param {Object} attributes Attributes of block.
 *
 * @returns {object} Modified props of save element.
 */
const styleWeightControl = (saveElementProps, blockType, attributes) => {
    // Do nothing if it's another block than our defined ones.
    if (!enableWeight.includes(blockType.name)) {
        return saveElementProps;
    }

    return saveElementProps

};

addFilter(
    'blocks.getSaveContent.extraProps',
    'register-block-control/get-save-content/extra-props',
    styleWeightControl
);