import assign from 'lodash.assign';
import classnames from 'classnames/dedupe'

const { addFilter } = wp.hooks;
const { __ } = wp.i18n;

const classPrefix = 'has-'

// Enable Weight control on the following blocks
const enableAlign = [
    'core/group',
];

// Available Weight control options
const maxWidthValues = [
    {
        label: __('Default', 'thor'),
        value: 'default',
    },
    {
        label: __('XS - 380px', 'thor'),
        value: 'xs',
    },
    {
        label: __('SM - 500px', 'thor'),
        value: 'sm',
    },
    {
        label: __('MD - 640px', 'thor'),
        value: 'md',
    },
    {
        label: __('ML - 768px', 'thor'),
        value: 'ml',
    },
    {
        label: __('LG - 800px', 'thor'),
        value: 'lg',
    },
    {
        label: __('XL - 1024px', 'thor'),
        value: 'xl',
    },
    {
        label: __('2XL - 1280px', 'thor'),
        value: '2xl',
    },
    {
        label: __('3XL - 1440px', 'thor'),
        value: '3xl',
    },
    {
        label: __('4XL - 1750px', 'thor'),
        value: '4xl',
    },
    {
        label: __('Container', 'thor'),
        value: 'container',
    },
];

const alignValues = [
   {
    label: __('Not selected', 'thor'),
    value: 'default',
  },
  {
    label: __('Left (initial)', 'thor'),
    value: 'left',
  },
  {
    label: __('Center', 'thor'),
    value: 'center',
  },
  {
    label: __('Right', 'thor'),
    value: 'right',
  },
];

/**
 * Add align control attribute to block.
 *
 * @param {object} settings Current block settings.
 * @param {string} name Name of block.
 *
 * @returns {object} Modified block settings.
 */
const alignControl = (settings, name) => {
    // Do nothing if it's another block than our defined ones.
    if (!enableAlign.includes(name)) {
        return settings;
    }

    if (name.includes('acf') || name.includes('core')) {
        settings.attributes = assign(settings.attributes, {
            align: {
                type: 'string',
                default: '',
            },
            maxWidth: {
                type: 'string',
                default: '',
            },
        });
    }
    return settings;

};

addFilter('blocks.registerBlockType', 'register-block-control/attribute/alignControl', alignControl);

const { createHigherOrderComponent } = wp.compose;
const { Fragment } = wp.element;
const { InspectorControls } = wp.blockEditor;
const { PanelBody, SelectControl } = wp.components;

/**
 * Create HOC to add align control to inspector controls of block.
 */
const extendalignControl = createHigherOrderComponent((BlockEdit) => {
    return ( props ) => {
        // Do nothing if it's another block than our defined ones.
        if ( ! enableAlign.includes( props.name ) ) {
            return (
                <BlockEdit { ...props } />
                );
            }
            
            
        const { align } = props.attributes;
        const { maxWidth } = props.attributes;

        let classNames = props.attributes.className

        alignValues.forEach(({ value}) => {
          classNames = classnames(classNames, { [`${classPrefix}${value}-alignment`]: align == value })
        })

        maxWidthValues.forEach(({ value}) => {
          classNames = classnames(classNames, { [`${classPrefix}${value}-max-width`]: maxWidth == value })
        })

        props.attributes.className = classNames



        return (
            <Fragment>
                <BlockEdit {...props} />
                <InspectorControls>
                    <PanelBody
                        title={__('Container options', 'thor')}
                        initialOpen={true}
                    >
                        <SelectControl
                            label={__('Alignment', 'thor')}
                            value={align}
                            options={alignValues}
                            onChange={(selectedalignOption) => {
                                props.setAttributes({
                                    align: selectedalignOption,
                                });
                            }}
                        />
                        <SelectControl
                            label={__('Max Width', 'thor')}
                            value={maxWidth}
                            options={maxWidthValues}
                            onChange={(selectedmaxWidthOption) => {
                                props.setAttributes({
                                    maxWidth: selectedmaxWidthOption,
                                });
                            }}
                        />
                    </PanelBody>
                </InspectorControls>
            </Fragment>
        );
    };
}, 'extendalignControl');

addFilter('editor.BlockEdit', 'extend-block-example/with-align-control', extendalignControl);



/**
 * Add margin style attribute to save element of block.
 *
 * @param {object} saveElementProps Props of save element.
 * @param {Object} blockType Block type information.
 * @param {Object} attributes Attributes of block.
 *
 * @returns {object} Modified props of save element.
 */
const styleWeightControl = (saveElementProps, blockType, attributes) => {
    // Do nothing if it's another block than our defined ones.
    if (!enableAlign.includes(blockType.name)) {
        return saveElementProps;
    }

    return saveElementProps

};

addFilter(
    'blocks.getSaveContent.extraProps',
    'register-block-control/get-save-content/extra-props',
    styleWeightControl
);