import assign from 'lodash.assign';
const { addFilter } = wp.hooks;
const { __ } = wp.i18n;

// Enable buttonWidth control on the following blocks
const enableButtonWidth = [
    'core/button',
];

const fontFamilyDefault = [
    {
        label: __( 'Inherit','muy-lindo' ),
        value: 'inherit',
    },
    {
        label: __( 'Primary','muy-lindo' ),
        value: 'font-primary',
    },
    {
        label: __( 'Secondary', 'muy-lindo' ),
        value: 'font-secondary',
    },
];

const fontWeightValues = [
    {
        label: __('Inherit', 'muy-lindo'),
        value: 'inherit',
    },
    {
        label: __('Thin 100', 'muy-lindo'),
        value: 'thin',
    },
    {
        label: __('Extra-light 200', 'muy-lindo'),
        value: 'extra-light',
    },
    {
        label: __('Light 300', 'muy-lindo'),
        value: 'light',
    },
    {
        label: __('Regular 400', 'muy-lindo'),
        value: 'regular',
    },
    {
        label: __('Medium 500', 'muy-lindo'),
        value: 'medium',
    },
    {
        label: __('Semi-bold 600', 'muy-lindo'),
        value: 'semi-bold',
    },
    {
        label: __('Bold 700', 'muy-lindo'),
        value: 'bold',
    },
    {
        label: __('Extra-bold 800', 'muy-lindo'),
        value: 'extra-bold',
    },
    {
        label: __( 'Black 900','muy-lindo' ),
        value: 'black',
    },
];

const fontSizesDefault = [
    {
        label: __('Default', 'muy-lindo'),
        value: 'default-font-size',
    },
    {
        label: __('Extra Small', 'muy-lindo'),
        value: 'extra-small-font-size',
    },
    {
        label: __('Small', 'muy-lindo'),
        value: 'small-font-size',
    },
    {
        label: __('Normal', 'muy-lindo'),
        value: 'medium-font-size',
    },
    {
        label: __('Large', 'muy-lindo'),
        value: 'large-font-size',
    },
    {
        label: __('Extra Large', 'muy-lindo'),
        value: 'xl-font-size',
    },
    {
        label: __('2XL', 'muy-lindo'),
        value: '2-xl-font-size',
    },
    {
        label: __('3XL', 'muy-lindo'),
        value: '3-xl-font-size',
    },
    {
        label: __('4XL', 'muy-lindo'),
        value: '4-xl-font-size',
    },
    {
        label: __('5XL', 'muy-lindo'),
        value: '5-xl-font-size',
    },
    {
        label: __( '6XL','muy-lindo' ),
        value: '6-xl-font-size',
    },
];


// Available buttonWidth control options
const buttonWidthDefault = [
    {
        label: __( 'Default','muy-lindo' ),
        value: 'default',
    },
    {
        label: __( 'Small','muy-lindo' ),
        value: 'small',
    },
    {
        label: __( 'Medium', 'muy-lindo' ),
        value: 'medium',
    },
    {
        label: __( 'Large', 'muy-lindo' ),
        value: 'large',
    },
    {
        label: __( 'Full Width', 'muy-lindo' ),
        value: 'full-width',
    },
];

const buttonHeightValues = [
    {
        label: __('Default', 'muy-lindo'),
        value: 'default',
    },
    {
        label: __('Short', 'muy-lindo'),
        value: 'short',
    },
    {
        label: __('Medium', 'muy-lindo'),
        value: 'medium',
    },
    {
        label: __('Tall', 'muy-lindo'),
        value: 'tall',
    },
];

/**
 * Add buttonWidth control attribute to block.
 *
 * @param {object} settings Current block settings.
 * @param {string} name Name of block.
 *
 * @returns {object} Modified block settings.
 */
const buttonControl = ( settings, name ) => {
    // Do nothing if it's another block than our defined ones.
    if ( ! enableButtonWidth.includes( name ) ) {
        return settings;
    }

    // Use Lodash's assign to gracefully handle if attributes are undefined
    settings.attributes = assign( settings.attributes, {
        buttonWidth: {
            type: 'string',
            default: buttonWidthDefault[0].value,
        },
        buttonHeight: {
            type: 'string',
            default: buttonHeightValues[0].value,
        },
        fontType: {
            type: 'string',
            default: fontFamilyDefault[0].value,
        },
        fontSizes: {
            type: 'string',
            default: fontSizesDefault[0].value,
        },
        fontWeight: {
            type: 'string',
            default: fontWeightValues[0].value,
        },
    } );
    return settings;

};

addFilter('blocks.registerBlockType', 'register-block-control/attribute/buttonControl', buttonControl);

const { createHigherOrderComponent } = wp.compose;
const { Fragment } = wp.element;
const { InspectorControls } = wp.blockEditor;
const { PanelBody, SelectControl } = wp.components;

/**
 * Create HOC to add buttonWidth control to inspector controls of block.
 */
const extendbuttonControl = createHigherOrderComponent( ( BlockEdit ) => {
    return ( props ) => {
        // Do nothing if it's another block than our defined ones.
        if ( ! enableButtonWidth.includes( props.name ) ) {
            return (
                <BlockEdit { ...props } />
                );
            }
            
            
        const { buttonWidth } = props.attributes;
        const { buttonHeight } = props.attributes;
        const { fontType } = props.attributes;
        const { fontSizes } = props.attributes;
        const { fontWeight } = props.attributes;
        var className = '';
        // add has-buttonWidth-xy class to block
        if ( buttonHeight ) {
            className += ` muy-button-height-${ buttonHeight }`;
        }
        if ( buttonWidth ) {
            className += ` muy-button-width-${ buttonWidth }`;
        }
        if ( fontWeight ) {
            className += ` muy-${ fontWeight }`;
        }
        if ( fontType ) {
            className += ` muy-${ fontType }`;
        }
        if ( fontSizes ) {
            className += ` has-${ fontSizes }`;
        }
        props.attributes.className = className;


        return (
            <Fragment>
                <BlockEdit { ...props } />
                <InspectorControls>
                    <PanelBody
                        title={ __( 'Buttons Control','muy-lindo' ) }
                        initialOpen={ true }
                    >
                        <SelectControl
                            label={ __( 'Width','muy-lindo' ) }
                            value={ buttonWidth }
                            options={ buttonWidthDefault }
                            onChange={ ( selectedbuttonWidthOption ) => {
                                props.setAttributes( {
                                    buttonWidth: selectedbuttonWidthOption,
                                } );
                            } }
                        />
                        <SelectControl
                            label={ __( 'Height','muy-lindo' ) }
                            value={ buttonHeight }
                            options={ buttonHeightValues }
                            onChange={ ( selectedbuttonHeightOption ) => {
                                props.setAttributes( {
                                    buttonHeight: selectedbuttonHeightOption,
                                } );
                            } }
                        />
                        <SelectControl
                            label={ __( 'Font','muy-lindo' ) }
                            value={ fontType }
                            options={ fontFamilyDefault }
                            onChange={ ( selectedfontTypeOption ) => {
                                props.setAttributes( {
                                    fontType: selectedfontTypeOption,
                                } );
                            } }
                        />
                        <SelectControl
                            label={ __( 'Font Size','muy-lindo' ) }
                            value={ fontSizes }
                            options={ fontSizesDefault }
                            onChange={ ( selectedfontSizesOption ) => {
                                props.setAttributes( {
                                    fontSizes: selectedfontSizesOption,
                                } );
                            } }
                        />
                        <SelectControl
                            label={ __( 'Font weight','muy-lindo' ) }
                            value={ fontWeight }
                            options={ fontWeightValues }
                            onChange={ ( selectedfontWeightOption ) => {
                                props.setAttributes( {
                                    fontWeight: selectedfontWeightOption,
                                } );
                            } }
                        />
                    </PanelBody>
                </InspectorControls>
            </Fragment>
        );
    };
}, 'extendbuttonControl' );

addFilter( 'editor.BlockEdit', 'extend-block-example/with-buttonWidth-control', extendbuttonControl );



/**
 * Add margin style attribute to save element of block.
 *
 * @param {object} saveElementProps Props of save element.
 * @param {Object} blockType Block type information.
 * @param {Object} attributes Attributes of block.
 *
 * @returns {object} Modified props of save element.
 */
const stylebuttonControl = (saveElementProps, blockType, attributes) => {
  // Do nothing if it's another block than our defined ones.
  if (!enableButtonWidth.includes(blockType.name)) {
    return saveElementProps;
  }

  const buttonWidth = {
    'small': '',
    'medium': '',
    'large': '',
    'full-width': '',
  };
  const definebuttonHeight = {
    'short': '',
    'medium': '',
    'tall': '',
  };

  const fontFamily = {
    'primary': '',
    'secondary': '',
  };
  
  const defineFontSizes = {
    'extra-small-font-size': '',
    'small-font-size': '',
    'large-font-size': '',
    'xl-font-size': '',
    '2xl-font-size': '',
    '3xl-font-size': '',
    '4xl-font-size': '',
    '5xl-font-size': '',
    '6xl-font-size': '',
  };

  const defineFontWeight = {
    'thin': '',
    'extra-light': '',
    'light': '',
    'regular': '',
    'medium': '',
    'semi-bold': '',
    'bold': '',
    'extra-bold': '',
    'black': '',
  };

  if (attributes.buttonWidth in buttonWidth) {
    // Use Lodash's assign to gracefully handle if attributes are undefined
    assign(saveElementProps, {
    });
  }
  if (attributes.buttonHeight in definebuttonHeight) {
    // Use Lodash's assign to gracefully handle if attributes are undefined
    assign(saveElementProps, {
    });
  }

  if (attributes.fontType in fontFamily) {
    // Use Lodash's assign to gracefully handle if attributes are undefined
    assign(saveElementProps, {
    });
  }
  if (attributes.fontSizes in defineFontSizes) {
    // Use Lodash's assign to gracefully handle if attributes are undefined
    assign(saveElementProps, {
    });
  }
  if (attributes.fontWeight in defineFontWeight) {
    // Use Lodash's assign to gracefully handle if attributes are undefined
    assign(saveElementProps, {
    });
  }

  return saveElementProps;

};

addFilter('blocks.getSaveContent.extraProps', 'register-block-control/get-save-content/extra-props', stylebuttonControl);
