import assign from 'lodash.assign';
const { addFilter } = wp.hooks;
const { __ } = wp.i18n;

// Enable fontType control on the following blocks
const enableColumnControls = [
    'core/heading',
    'core/paragraph',
    'core/quote',
    'core/list',
];

// Available fontType control options
const fontFamilyDefault = [
    {
        label: __( 'Inherit','muy-lindo' ),
        value: 'inherit',
    },
    {
        label: __( 'Primary','muy-lindo' ),
        value: 'font-primary',
    },
    {
        label: __( 'Secondary', 'muy-lindo' ),
        value: 'font-secondary',
    },
];

const fontWeightValues = [
    {
        label: __('Inherit', 'muy-lindo'),
        value: 'inherit',
    },
    {
        label: __('Thin 100', 'muy-lindo'),
        value: 'thin',
    },
    {
        label: __('Extra-light 200', 'muy-lindo'),
        value: 'extra-light',
    },
    {
        label: __('Light 300', 'muy-lindo'),
        value: 'light',
    },
    {
        label: __('Regular 400', 'muy-lindo'),
        value: 'regular',
    },
    {
        label: __('Medium 500', 'muy-lindo'),
        value: 'medium',
    },
    {
        label: __('Semi-bold 600', 'muy-lindo'),
        value: 'semi-bold',
    },
    {
        label: __('Bold 700', 'muy-lindo'),
        value: 'bold',
    },
    {
        label: __('Extra-bold 800', 'muy-lindo'),
        value: 'extra-bold',
    },
    {
        label: __( 'Black 900','muy-lindo' ),
        value: 'black',
    },
];

/**
 * Add fontType control attribute to block.
 *
 * @param {object} settings Current block settings.
 * @param {string} name Name of block.
 *
 * @returns {object} Modified block settings.
 */
const columnsControl = ( settings, name ) => {
    // Do nothing if it's another block than our defined ones.
    if ( ! enableColumnControls.includes( name ) ) {
        return settings;
    }

    // Use Lodash's assign to gracefully handle if attributes are undefined
    settings.attributes = assign( settings.attributes, {
        fontType: {
            type: 'string',
            default: fontFamilyDefault[0].value,
        },
        fontWeight: {
            type: 'string',
            default: fontWeightValues[0].value,
        },
    } );
    return settings;

};

addFilter('blocks.registerBlockType', 'register-block-control/attribute/columnsControl', columnsControl);

const { createHigherOrderComponent } = wp.compose;
const { Fragment } = wp.element;
const { InspectorControls } = wp.blockEditor;
const { PanelBody, SelectControl } = wp.components;

/**
 * Create HOC to add fontType control to inspector controls of block.
 */
const extendColumnsControl = createHigherOrderComponent( ( BlockEdit ) => {
    return ( props ) => {
        // Do nothing if it's another block than our defined ones.
        if ( ! enableColumnControls.includes( props.name ) ) {
            return (
                <BlockEdit { ...props } />
                );
            }
            
            
        const { fontType } = props.attributes;
        const { fontWeight } = props.attributes;
        var className = '';
        // add has-fontType-xy class to block
        if ( fontWeight ) {
            className += ` muy-${ fontWeight }`;
        }
        if ( fontType ) {
            className += ` muy-${ fontType }`;
        }
        props.attributes.className = className;


        return (
            <Fragment>
                <BlockEdit { ...props } />
                <InspectorControls>
                    <PanelBody
                        title={ __( 'Font control','muy-lindo' ) }
                        initialOpen={ true }
                    >
                        <SelectControl
                            label={ __( 'Font','muy-lindo' ) }
                            value={ fontType }
                            options={ fontFamilyDefault }
                            onChange={ ( selectedfontTypeOption ) => {
                                props.setAttributes( {
                                    fontType: selectedfontTypeOption,
                                } );
                            } }
                        />
                        <SelectControl
                            label={ __( 'Font weight','muy-lindo' ) }
                            value={ fontWeight }
                            options={ fontWeightValues }
                            onChange={ ( selectedfontWeightOption ) => {
                                props.setAttributes( {
                                    fontWeight: selectedfontWeightOption,
                                } );
                            } }
                        />
                    </PanelBody>
                </InspectorControls>
            </Fragment>
        );
    };
}, 'extendColumnsControl' );

addFilter( 'editor.BlockEdit', 'extend-block-example/with-fontType-control', extendColumnsControl );



/**
 * Add margin style attribute to save element of block.
 *
 * @param {object} saveElementProps Props of save element.
 * @param {Object} blockType Block type information.
 * @param {Object} attributes Attributes of block.
 *
 * @returns {object} Modified props of save element.
 */
const styleColumnsControl = (saveElementProps, blockType, attributes) => {
  // Do nothing if it's another block than our defined ones.
  if (!enableColumnControls.includes(blockType.name)) {
    return saveElementProps;
  }

  const fontFamily = {
    'primary': '',
    'secondary': '',
  };
  const defineFontWeight = {
    'thin': '',
    'extra-light': '',
    'light': '',
    'regular': '',
    'medium': '',
    'semi-bold': '',
    'bold': '',
    'extra-bold': '',
    'black': '',
  };

  if (attributes.fontType in fontFamily) {
    // Use Lodash's assign to gracefully handle if attributes are undefined
    assign(saveElementProps, {
    });
  }
  if (attributes.fontWeight in defineFontWeight) {
    // Use Lodash's assign to gracefully handle if attributes are undefined
    assign(saveElementProps, {
    });
  }

  return saveElementProps;

};

addFilter('blocks.getSaveContent.extraProps', 'register-block-control/get-save-content/extra-props', styleColumnsControl);
