import assign from 'lodash.assign';
const { addFilter } = wp.hooks;
const { __ } = wp.i18n;

// Enable maxWidth control on the following blocks
const enableMaxWidth = [
    'core/group',
];

// Available maxWidth control options
const maxWidthValues = [
    {
        label: __( 'Default','muy-lindo' ),
        value: 'default',
    },
    {
        label: __( 'XS - 380px','muy-lindo' ),
        value: 'xs',
    },
    {
        label: __( 'SM - 500px', 'muy-lindo' ),
        value: 'sm',
    },
    {
        label: __( 'MD - 650px', 'muy-lindo' ),
        value: 'md',
    },
    {
        label: __( 'LG - 768px', 'muy-lindo' ),
        value: 'lg',
    },
    {
        label: __( 'XL - 1024px', 'muy-lindo' ),
        value: 'xl',
    },
    {
        label: __( '2XL - 1280px', 'muy-lindo' ),
        value: '2xl',
    },
    {
        label: __( '3XL - 1440px', 'muy-lindo' ),
        value: '3xl',
    },
];

/**
 * Add maxWidth control attribute to block.
 *
 * @param {object} settings Current block settings.
 * @param {string} name Name of block.
 *
 * @returns {object} Modified block settings.
 */
const maxWidthControl = ( settings, name ) => {
    // Do nothing if it's another block than our defined ones.
        if (!enableMaxWidth.includes(name)) {
          return settings;
        }

    // Use Lodash's assign to gracefully handle if attributes are undefined
    settings.attributes = assign( settings.attributes, {
        maxWidth: {
            type: 'string',
            default: maxWidthValues[ 0 ].value,
        },
    } );
    return settings;

};

addFilter('blocks.registerBlockType', 'register-block-control/attribute/maxWidthControl', maxWidthControl);

const { createHigherOrderComponent } = wp.compose;
const { Fragment } = wp.element;
const { InspectorControls } = wp.blockEditor;
const { PanelBody, SelectControl } = wp.components;

/**
 * Create HOC to add maxWidth control to inspector controls of block.
 */
const extendMaxWidthControl = createHigherOrderComponent( ( BlockEdit ) => {
    return ( props ) => {
        // Do nothing if it's another block than our defined ones.
        if ( ! enableMaxWidth.includes( props.name ) ) {
            return (
                <BlockEdit { ...props } />
            );
        }
            
        const { maxWidth } = props.attributes;
        var className = '';
        // add has-maxWidth-xy class to block
        if ( maxWidth ) {
            className += ` muy-maxwidth-${ maxWidth }`;
        }
        props.attributes.className = className;


        return (
            <Fragment>
                <BlockEdit { ...props } />
                <InspectorControls>
                    <PanelBody
                        title={ __( 'Max Width Control','muy-lindo' ) }
                        initialOpen={ true }
                    >
                        <SelectControl
                            label={ __( 'Max Width','muy-lindo' ) }
                            value={ maxWidth }
                            options={ maxWidthValues }
                            onChange={ ( selectedmaxWidthOption ) => {
                                props.setAttributes( {
                                    maxWidth: selectedmaxWidthOption,
                                } );
                            } }
                        />
                    </PanelBody>
                </InspectorControls>
            </Fragment>
        );
    };
}, 'extendMaxWidthControl' );

addFilter( 'editor.BlockEdit', 'extend-block-example/with-maxWidth-control', extendMaxWidthControl );



/**
 * Add margin style attribute to save element of block.
 *
 * @param {object} saveElementProps Props of save element.
 * @param {Object} blockType Block type information.
 * @param {Object} attributes Attributes of block.
 *
 * @returns {object} Modified props of save element.
 */
const stylemaxWidthControl = (saveElementProps, blockType, attributes) => {
  // Do nothing if it's another block than our defined ones.
  if (!enableMaxWidth.includes(blockType.name)) {
    return saveElementProps;
  }

  const maxwidth = {
    'xs': '',
    'sm': '',
    'md': '',
    'lg': '',
    'xl': '',
    '2xl': '',
    '3xl': '',
  };

  if (attributes.maxWidth in maxwidth) {
    // Use Lodash's assign to gracefully handle if attributes are undefined
    assign(saveElementProps, {
    });
  }

  return saveElementProps;

};

addFilter('blocks.getSaveContent.extraProps', 'register-block-control/get-save-content/extra-props', stylemaxWidthControl);
