import assign from 'lodash.assign';

const { addFilter } = wp.hooks;
const { __ } = wp.i18n;

// Enable columnsInvert control on the following blocks
const enableColumnControls = [
    'core/columns',
];

// Available columnsInvert control options
const invertColumnsValues = [
    {
        label: __('Not Inverted', 'muy-lindo'),
        value: 'not-reverted',
    },
    {
        label: __('Inverted', 'muy-lindo'),
        value: 'revert',
    },
];

const tabletOneColumnValues = [
    {
        label: __('Two Columns', 'muy-lindo'),
        value: 'tablet-two-columns',
    },
    {
        label: __('One Column', 'muy-lindo'),
        value: 'tablet-one-column',
    },
];

/**
 * Add columnsInvert control attribute to block.
 *
 * @param {object} settings Current block settings.
 * @param {string} name Name of block.
 *
 * @returns {object} Modified block settings.
 */
const columnsControl = (settings, name) => {
    // Do nothing if it's another block than our defined ones.
    if (!enableColumnControls.includes(name)) {
        return settings;
    }

    // Use Lodash's assign to gracefully handle if attributes are undefined
    settings.attributes = assign(settings.attributes, {
        columnsInvert: {
            type: 'string',
            default: invertColumnsValues[0].value,
        },
        tabletOneColumn: {
            type: 'string',
            default: tabletOneColumnValues[0].value,
        },
    });
    return settings;

};

addFilter('blocks.registerBlockType', 'register-block-control/attribute/columnsControl', columnsControl);

const { createHigherOrderComponent } = wp.compose;
const { Fragment } = wp.element;
const { InspectorControls } = wp.blockEditor;
const { PanelBody, SelectControl } = wp.components;

/**
 * Create HOC to add columnsInvert control to inspector controls of block.
 */
const extendColumnsControl = createHigherOrderComponent((BlockEdit) => {
    return (props) => {
        // Do nothing if it's another block than our defined ones.
        if (!enableColumnControls.includes(props.name)) {
            return (
                <BlockEdit {...props} />
            );
        }
            
            
        const { columnsInvert } = props.attributes;
        const { tabletOneColumn } = props.attributes;
        var className = '';
        // add has-columnsInvert-xy class to block
        if (tabletOneColumn) {
            className += ` muy-${tabletOneColumn}`;
        }
        if (columnsInvert) {
            className += ` muy-${columnsInvert}`;
        }
        props.attributes.className = className;


        return (
            <Fragment>
                <BlockEdit {...props} />
                <InspectorControls>
                    <PanelBody
                        title={__('Responsive Columns', 'muy-lindo')}
                        initialOpen={true}
                    >
                        <SelectControl
                            label={__('Invert columns', 'muy-lindo')}
                            value={columnsInvert}
                            options={invertColumnsValues}
                            onChange={(selectedcolumnsInvertOption) => {
                                props.setAttributes({
                                    columnsInvert: selectedcolumnsInvertOption,
                                });
                            }}
                        />
                        <SelectControl
                            label={__('Tablet view', 'muy-lindo')}
                            value={tabletOneColumn}
                            options={tabletOneColumnValues}
                            onChange={(selectedTabletOneColumnOption) => {
                                props.setAttributes({
                                    tabletOneColumn: selectedTabletOneColumnOption,
                                });
                            }}
                        />
                    </PanelBody>
                </InspectorControls>
            </Fragment>
        );
    };
}, 'extendColumnsControl');

addFilter('editor.BlockEdit', 'extend-block-example/with-columnsInvert-control', extendColumnsControl);



/**
 * Add margin style attribute to save element of block.
 *
 * @param {object} saveElementProps Props of save element.
 * @param {Object} blockType Block type information.
 * @param {Object} attributes Attributes of block.
 *
 * @returns {object} Modified props of save element.
 */
const styleColumnsControl = (saveElementProps, blockType, attributes) => {
    // Do nothing if it's another block than our defined ones.
    if (!enableColumnControls.includes(blockType.name)) {
        return saveElementProps;
    }

    const margins = {
        'not-reverted': '',
        'revert': '',
    };
    const tabletView = {
        'tablet-two-columns': '',
        'tablet-one-columns': '',
    };

    if (attributes.columnsInvert in margins) {
        // Use Lodash's assign to gracefully handle if attributes are undefined
        assign(saveElementProps, {
        });
    }
    if (attributes.tabletOneColumn in tabletView) {
        // Use Lodash's assign to gracefully handle if attributes are undefined
        assign(saveElementProps, {
        });
    }

    return saveElementProps;

};

addFilter('blocks.getSaveContent.extraProps', 'register-block-control/get-save-content/extra-props', styleColumnsControl);
